# SafeNet Access Exchange High Availability Cluster - Podman for RHEL 9

This project sets up a High Availability (HA) cluster for SafeNet Access Exchange (SAE) using Podman on Red Hat Enterprise Linux 9. The setup consists of three SAE nodes and an Nginx load balancer, integrating with a MySQL backend via ProxySQL.

## Project Structure

```
SAE_HA_Cluster_Podman/
├── README.md                                  # This documentation file
├── .env.example                               # Template for environment variables
├── podman-compose-node1.yml                   # Node 1 Config (SAE + NGINX)
├── podman-compose-node2.yml                   # Node 2 Config (SAE Worker)
├── podman-compose-node3.yml                   # Node 3 Config (SAE Worker)
├── cache-ispn-jdbc-ping-tcp.xml               # Infinispan cache configuration
├── nginx.conf                                 # Nginx Load Balancer configuration
└── certs/                                     # SSL Certificates
    ├── cert.pem                               # Public Certificate
    └── privkey.pem                            # Private Key
```

## Services Breakdown

### Node 1
*   **`sae-node1`**: SafeNet Access Exchange (SAE) instance.
*   **`nginx-lb`**: Nginx Reverse Proxy / Load Balancer (Entry point).

### Node 2
*   **`sae-node2`**: SafeNet Access Exchange (SAE) instance.

### Node 3
*   **`sae-node3`**: SafeNet Access Exchange (SAE) instance.

## Architecture

The cluster is distributed across three nodes. All nodes communicate via JGroups for cache replication using the host network.

*   **Node 1**: Hosts the Load Balancer and SAE Node 1.
*   **Node 2**: SAE Node 2.
*   **Node 3**: SAE Node 3.

## Prerequisites

*   Podman and podman-compose installed on all nodes.
*   Firewall rules allowing traffic on ports:
    *   `80`, `443` (Nginx on Node 1, external access)
    *   `7800` (JGroups, internal cluster communication between all nodes)
    *   `8080` (SAE, internal access from LB on Node 1)

Note: This deployment assumes that a highly available MySQL database is already deployed, either as a managed service from a cloud provider or by following the instructions in the "MySQL Cluster Deployment (for High-Availability)" section. You will need to update the KC_DB_URL environment variable in your .env file with the connection string for your database.

### Install Podman

```bash
# Install Podman and podman-compose
sudo dnf install -y podman podman-compose

# Verify installation
podman --version
podman-compose --version
```

### Configure Firewall

On **all nodes**:
```bash
# Open required ports
sudo firewall-cmd --permanent --add-port=8080/tcp
sudo firewall-cmd --permanent --add-port=7800/tcp
sudo firewall-cmd --reload
```

On **Node 1** only:
```bash
# Open HTTP/HTTPS ports
sudo firewall-cmd --permanent --add-port=80/tcp
sudo firewall-cmd --permanent --add-port=443/tcp
sudo firewall-cmd --reload

# Verify ports
sudo firewall-cmd --list-ports
```

### Configure SELinux

On **all nodes**:
```bash
# Enable container booleans
sudo setsebool -P container_manage_cgroup on
sudo setsebool -P virt_sandbox_use_all_caps on

# After copying project files, restore SELinux context
cd /path/to/SAE_HA_Cluster_Podman
sudo restorecon -Rv .

# Verify
getenforce
```

## Deployment Steps

Follow these steps to deploy the complete solution.

### Step 1: Configure Environment Variables
1.  On **all nodes**, create a `.env` file from the example:
    ```bash
    cp .env.example .env
    ```
2.  Edit `.env` and set the following:
    *   **Passwords**: Set strong passwords for `MYSQL_PASSWORD`, `MYSQL_ROOT_PASSWORD`, and `KEYCLOAK_ADMIN_PASSWORD`.
    *   **Node IPs**: Update `SAE_NODE1_IP`, `SAE_NODE2_IP`, and `SAE_NODE3_IP` with the actual IP addresses of your servers.
    *   **Hostname**: Ensure `KC_HOSTNAME` matches your domain name.
    *   **Database URL**: Update `KC_DB_URL` to point to your ProxySQL instance.

### Step 2: Configure SSL Certificates
1.  On **Node 1** (where Nginx runs), navigate to the project directory.
2.  Place your valid SSL certificate and private key in the `certs/` folder:
    *   Replace `certs/cert.pem` with your public certificate.
    *   Replace `certs/privkey.pem` with your private key.

### Step 3: Configure Nginx
1.  Open `nginx.conf` on **Node 1**.
2.  Replace the placeholders `<SAE_NODE1_IP>`, `<SAE_NODE2_IP>`, `<SAE_NODE3_IP>`, and `<SAE_HOST_NAME>` with the actual IP addresses of your nodes and domain name (same as `KC_HOSTNAME` in .env file).

### Step 4: Load SAE Container Image
On **all nodes**:
```bash
# Load the SAE image
podman load -i /path/to/safenet-access-exchange-latest.tar

# Verify
podman images | grep safenet
```

### Step 5: Deploy Node 1 (Primary)
On **Node 1**, run:
```bash
podman compose -f podman-compose-node1.yml up -d
```
*   Verify that Nginx and `sae-node1` are running.
*   Check logs: `podman logs -f sae-node1`

### Step 6: Deploy Worker Nodes
On **Node 2**, run:
```bash
podman compose -f podman-compose-node2.yml up -d
```

On **Node 3**, run:
```bash
podman compose -f podman-compose-node3.yml up -d
```

### Step 7: Verification
1.  Access the application via the configured hostname (e.g., `https://sae.example.com`).
2.  Check logs to ensure nodes have discovered each other via JGroups:
    ```bash
    podman logs sae-node1 | grep -i "jgroups\|cluster"
    ```

## Managing the Cluster

### Start/Stop Containers
```bash
# Start
podman compose -f podman-compose-node1.yml up -d
podman compose -f podman-compose-node2.yml up -d
podman compose -f podman-compose-node3.yml up -d

# Stop
podman compose -f podman-compose-node1.yml down
podman compose -f podman-compose-node2.yml down
podman compose -f podman-compose-node3.yml down

# View logs
podman logs -f sae-node1
podman logs -f nginx-lb
```

## Troubleshooting

### SELinux Issues
```bash
# Check for denials
sudo ausearch -m avc -ts recent

# Fix context
sudo restorecon -Rv /path/to/SAE_HA_Cluster_Podman
```

### Port Connectivity
```bash
# Test connectivity between nodes
nc -zv <NODE_IP> 7800
nc -zv <NODE_IP> 8080
```

## Security Notes

*   **Firewall**: Ensure your OS firewall blocks external access to ports 7800 and 8080.
*   **SELinux**: Keep SELinux in enforcing mode for production.
*   **Secrets**: Keep your `.env` file secure.
*   **SSL**: Ensure your certificates are valid and trusted.
